(* ::Package:: *)

(* Copyright 1997-2024 by California Inst. of Technology, Pasadena, CA. *)

(* Only call this package file from CurveFit.m *)
(* The code assumes that we are already in the CurveFit` context. *)

(* Plots.m - Data plotting routines *)

(* version 1.99g  4/2024: Fixed a residual plot bug when fY[x] is an interpolation *)
                           
(* 1/2016: made default error bar thickness Medium vice Thin *)
(* 3/2018: mod to ScatterDataPlot aspect ratio *)



(***********************************************************)
(* Plot Option usage messages *)

(*---------------------------------------------------------*)
(* CurveFit-set default values for several plot options *)

$AspectRatio::usage =
 "\!\(\*StyleBox[\"$AspectRatio\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"AspectRatio\",\nFontFamily->\"Courier\"]\) "<>
 "used by all types of plots. "<>
 "Change the value of this variable to adjust the default "<>
 "plot (height/width) ratio. It is initialized with the Mathematica "<>
 "default value of "<>
 "\!\(\*StyleBox[\"1/GoldenRatio\",\nFontFamily->\"Courier\"]\) "<>
 "when the CurveFit package is loaded."<>
 "\n\n"<>
 "See also: AspectRatio";

$ImageSize::usage =
 "\!\(\*StyleBox[\"$ImageSize\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"ImageSize\",\nFontFamily->\"Courier\"]\) "<>
 "used by all types of plots. "<>
 "Change the value of this variable to adjust the default plot size. "<>
 "It is initialized with a value of 500 (i.e., width = 500 points) "<>
 "when the CurveFit package is loaded."<>
 "\n\n"<>
 "See also: ImageSize";

$LabelStyle::usage =
 "\!\(\*StyleBox[\"$LabelStyle\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"LabelStyle\",\nFontFamily->\"Courier\"]\) "<>
 "used by all types of plots. "<>
 "Change the value of this variable to adjust the default "<>
 "plot label style. When the CurveFit package is loaded "<>
 "it is initialized to \n\t\t"<>
 "\!\(\*StyleBox[\"Directive[Italic, FontFamily->\",\nFontFamily->\"Courier\"]\)\""<>
 "\!\(\*StyleBox[\"Helvetica\",\nFontFamily->\"Courier\"]\)\""<>
 "\!\(\*StyleBox[\"]\",\nFontFamily->\"Courier\"]\) "<>
 "\n\n"<>
 "See also: LabelStyle";

$GridLines::usage = 
 "\!\(\*StyleBox[\"$GridLines\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"GridLines\",\nFontFamily->\"Courier\"]\) "<>
 "used by all types of plots. "<>
 "Change the value of this variable to adjust the default "<>
 "grid line display behavior.\n\n"<>
 "See also: GridLines";

$GridLinesStyle::usage = 
 "\!\(\*StyleBox[\"$GridLinesStyle\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"GridLinesStyle\",\nFontFamily->\"Courier\"]\) "<>
 "used by all types of plots. "<>
 "Change the value of this variable to adjust the default "<>
 "grid line color, thickness, and/or line style.\n\n"<>
 "See also: GridLinesStyle";

$PlotStyle::usage = 
 "\!\(\*StyleBox[\"$PlotStyle\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"PlotStyle\",\nFontFamily->\"Courier\"]\) "<>
 "used by standard Mathematica plots such as "<>
 "\!\(\*StyleBox[\"Plot\",\nFontFamily->\"Courier\"]\) "<>
 "and "<>
 "\!\(\*StyleBox[\"ListPlot\",\nFontFamily->\"Courier\"]\) "<>
 "(but not custom CurveFit plots such as "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 ", whose "<>
 "default styles are held in "<>
 "\!\(\*StyleBox[\"$DataPlotStyle\",\nFontFamily->\"Courier\"]\)). "<>
 "Change the value of this variable to adjust the default "<>
 "plot lines' colors, thicknesses, etc. "<>
 "Its value is set to "<>
 "\!\(\*StyleBox[\"Automatic\",\nFontFamily->\"Courier\"]\) "<>
 "at "<>
 "initialization.\n\n"<>
 "See also: PlotStyle, $DataPlotStyle";

$DataPlotStyle::usage = 
 "\!\(\*StyleBox[\"$DataPlotStyle\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"PlotStyle\",\nFontFamily->\"Courier\"]\) "<>
 "used by CurveFit's data and fit plot functions such as "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\) "<>
 "and "<>
 "\!\(\*StyleBox[\"LogDifferencePlot\",\nFontFamily->\"Courier\"]\). "<>
 "Change the value of this variable to adjust the default "<>
 "data plots' line colors, thicknesses, etc.\n\n"<>
 "See also: PlotStyle, $PlotStyle";

$PlotMarkers::usage = 
 "\!\(\*StyleBox[\"$PlotMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"PlotMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "used by standard Mathematica plots such as "<>
 "\!\(\*StyleBox[\"ListPlot\",\nFontFamily->\"Courier\"]\) "<>
 "(but not custom CurveFit plots such as "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 ", whose "<>
 "default styles are held in "<>
 "\!\(\*StyleBox[\"$DataPlotMarkers\",\nFontFamily->\"Courier\"]\)). "<>
 "Change the value of this variable to adjust the "<>
 "data point plot markers' default shapes, colors, sizes, etc., "<>
 "used by standard Mathematica plotting functions. "<>
 "Its value is set to "<>
 "\!\(\*StyleBox[\"Automatic\",\nFontFamily->\"Courier\"]\) "<>
 "at "<>
 "initialization.\n\n"<>
 "See also: PlotMarkers, $DataPlotMarkers";

$DataPlotMarkers::usage = 
 "\!\(\*StyleBox[\"$DataPlotMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"PlotMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "used by CurveFit's data and fit plot functions such as "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\) "<>
 "and "<>
 "\!\(\*StyleBox[\"LogDifferencePlot\",\nFontFamily->\"Courier\"]\). "<>
 "Change the value of this variable to adjust the "<>
 "data point plot markers' default shapes, colors, sizes, etc.\n\n"<>
 "See also: PlotMarkers, $PlotMarkers";

$IntervalMarkers::usage = 
 "\!\(\*StyleBox[\"$IntervalMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"IntervalMarkers\",\nFontFamily->\"Courier\"]\) "<>
 "used by both standard Mathematica as well as CurveFit data plots. "<>
 "Change the value of this variable to adjust the "<>
 "default way error bars are displayed.\n\n"<>
 "See also: IntervalMarkers";

$IntervalMarkersStyle::usage = 
 "\!\(\*StyleBox[\"$IntervalMarkersStyle\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the graphics option "<>
 "\!\(\*StyleBox[\"IntervalMarkersStyle\",\nFontFamily->\"Courier\"]\) "<>
 "used by both standard Mathematica as well as CurveFit data plots. "<>
 "Change the value of this variable to adjust "<>
 "error bars' default line colors, thicknesses, etc.\n\n"<>
 "See also: IntervalMarkersStyle";

(*---------------------------------------------------------*)
(* CurveFit plot options *)

DataSizeThreshold::usage = 
 "Plots of large data sets are generated slowly "<>
 "if the data point marker style is not simple, error bars must be "<>
 "drawn, and individual data point Tooltips are used. "<>
 "CurveFit data plotting functions such as "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\) "<>
 "include "<>
 "the additional, custom option "<>
 "\!\(\*StyleBox[\"DataSizeThreshold\",\nFontFamily->\"Courier\"]\) "<>
 "to help speed up the rendering of large data sets.\n\n"<>
 "\!\(\*StyleBox[\"DataSizeThreshold\",\nFontFamily->\"Courier\"]\) "<>
 "must be provided a list of two integers, e.g.:\n"<>
 "\t "<>
 "\!\(\*StyleBox[\"DataSizeThreshold -> {120, 300}\",\nFontFamily->\"Courier\"]\) "<>
 "\n\n"<>
 "The first (smaller) number sets the maximum size of a data set that "<>
 "will include full rendering of all data points, error bars, and Tooltips. "<>
 "Sets larger than this are \"large,\" and CurveFit plot functions will simplify "<>
 "their rendering of data points by setting the option:\n"<>
 "\t "<>
 "\!\(\*StyleBox[\"PlotMarkers -> None\",\nFontFamily->\"Courier\"]\)"<>
 "\n\n"<>
 "The second number sets the size threshold for \"very large\" data sets. "<>
 "Sets larger than this will exclude error bars and tooltips and will join the "<>
 "points into a single curve by setting the options:\n"<>
 "\t "<>
 "\!\(\*StyleBox[\"{IntervalMarkers -> None, Tips -> None, Joined -> True}"<>
 "\",\nFontFamily->\"Courier\"]\)"<>
 "\n\n"<>
 "Setting  "<>
 "\!\(\*StyleBox[\"DataSizeThreshold -> {0, Infinity}\",\nFontFamily->\"Courier\"]\) "<>
 "makes every data set \"large,\" but none is \"very large.\"  "<>
 "\!\(\*StyleBox[\"DataSizeThreshold -> {Infinity, Infinity}\",\nFontFamily->\"Courier\"]\) "<>
 "ensures that every data set is fully rendered regardless of size. "<>
 "The default pair of threshold values is held in $DataSizeThreshold.\n\n"<>
 "Any of these actions may be overridden by explicitly providing an option "<>
 "rule argument with the data plotting function call, e.g. "<>
 "\!\(\*StyleBox[\"LinearDataPlot[Tips -> All]\",\nFontFamily->\"Courier\"]\) "<>
 "will include individual data point Tooltips regardless of the data set size.\n\n"<>
 "See also:\n$DataSizeThreshold, PlotMarkers, $DataPlotMarkers, IntervalMarkers, Tips.";

$DataSizeThreshold::usage = 
 "\!\(\*StyleBox[\"$DataSizeThreshold\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the CurveFit plot option "<>
 "\!\(\*StyleBox[\"DataSizeThreshold\",\nFontFamily->\"Courier\"]\). "<>
 "Its value must be set to a list of two integers; at initialization it "<>
 "is set to "<>
 "\!\(\*StyleBox[\"{120, 300}\",\nFontFamily->\"Courier\"]\)."<>
 "\n\n"<>
 "See the information for "<>
 "\!\(\*StyleBox[\"DataSizeThreshold\",\nFontFamily->\"Courier\"]\) "<>
 "for details regarding the interpretation of these numbers.\n\n"<>
 "See also: DataSizeThreshold";

Tips::usage = 
 "\!\(\*StyleBox[\"Tips\",\nFontFamily->\"Courier\"]\):\n "<>
 "An option for CurveFit data and fit plots ("<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\), "<>
 "\!\(\*StyleBox[\"LogDifferencePlot\",\nFontFamily->\"Courier\"]\), "<>
 "etc.) controlling the display of Tooltips "<>
 "identifying the individual data points. The information displayed "<>
 "for a data point includes its index number in the data set list and its "<>
 "(\!\(\* StyleBox[\"x\",\nFontSlant->\"Italic\"]\), "<>
 "\!\(\* StyleBox[\"y\",\nFontSlant->\"Italic\"]\)) "<>
 "values and their uncertainties.\n"<>
 "\t\t "<>
 "\!\(\*StyleBox[\"Tips \[RightArrow] All    \t  \",\nFontFamily->\"Courier\"]\)"<>
 ": show tooltips regardless of the size of the data set. \n"<>
 "\t\t "<>
 "\!\(\*StyleBox[\"Tips \[RightArrow] None  \t  \",\nFontFamily->\"Courier\"]\)"<>
 ": do not show tooltips. \n"<>
 "\t\t "<>
 "\!\(\*StyleBox[\"Tips \[RightArrow] Automatic  \",\nFontFamily->\"Courier\"]\)"<>
 ": show tooltips if the data set is not too large. \n"<>
 "\n"<>
 "The default value for this option is stored in "<>
 "\!\(\*StyleBox[\"$Tips\",\nFontFamily->\"Courier\"]\), "<>
 "set to "<>
 "\!\(\*StyleBox[\"Automatic\",\nFontFamily->\"Courier\"]\) "<>
 "at "<>
 "initialization. The option "<>
 "\!\(\*StyleBox[\"DataSizeThreshold\",\nFontFamily->\"Courier\"]\) "<>
 "determines the cutoff set size.\n"<>
 "\n"<>
 "See also: $Tips, DataSizeThreshold";

$Tips::usage = 
 "\!\(\*StyleBox[\"$Tips\",\nFontFamily->\"Courier\"]\) "<>
 "holds the default value for the CurveFit plot option "<>
 "\!\(\*StyleBox[\"Tips\",\nFontFamily->\"Courier\"]\). "<>
 "At initialization it is set to "<>
 "\!\(\*StyleBox[\"Automatic\",\nFontFamily->\"Courier\"]\). "<>
 "Other choices include "<>
 "\!\(\*StyleBox[\"All\",\nFontFamily->\"Courier\"]\) and "<>
 "\!\(\*StyleBox[\"None\",\nFontFamily->\"Courier\"]\). "<>
 "See the information for "<>
 "\!\(\*StyleBox[\"Tips\",\nFontFamily->\"Courier\"]\) "<>
 "for details regarding the interpretation of these values.\n\n"<>
 "See also: Tips";




(***********************************************************)
(* Default option values for unique CurveFit plotting function options *)

$Tips = Automatic; (* Display of Tooltips is controlled by DataSizeThreshold *)
$DataSizeThreshold = {120,300}; (* The default size threshold values *)

(* Default option values for both Mathematica and CurveFit plotting functions *)
$AspectRatio = 1/GoldenRatio;
$GridLines = Automatic;
$GridLinesStyle = GrayLevel[.85];
$ImageSize = 500;
$LabelStyle = Directive[Italic, FontFamily->"Helvetica"];
$IntervalMarkers = "Fences";
$IntervalMarkersStyle = Directive[Thin,Black,Dashing[{}]];

(* Default option values for standard Mathematica Plot and ListPlot functions *)
$PlotStyle = Automatic;
$PlotMarkers = Automatic;

(* Default option values for CurveFit data set plotting functions *)
$DataPlotStyle = 
 { 
 (* 
    Each list element is enclosed in its own { } so that the list is interpreted 
    as a sequence of styles used for multiple data sets. This way a single data set
    uses only the first element of the list as its style description, and not the
    entire sequence of directives.
 *)
  {Directive[Thick, Darker[Red], Dashing[{}]]}, (* a single data set uses this one *)
  {Directive[Thick, Darker[Blue], Dashing[{}]]},
  {Directive[Thick, Darker[Green], Dashing[{}]]},
  {Directive[Thick, Purple, Dashing[{}]]}
 };
$DataPlotMarkers = 
 {
 (* 
    Unlike the case for PlotStyle, list elements are complete Graphics objects and
    not simply a sequence of graphics directives. In the case of a single data set,
    only the first element is used. Each element is a { }-enclosed pair because the
    second element is a factor by which the graphics object is scaled before use.
 *)
  {Graphics[{EdgeForm[{RGBColor[2/3, 0, 0]}], FaceForm[RGBColor[1, 1, 0]], 
   Disk[{0, 0}]}], 0.035},
  {Graphics[{EdgeForm[{Darker[Blue]}], FaceForm[Darker[Green]], 
   Rectangle[{-1,-1},{1,1}]}], 0.030},
  {Graphics[{EdgeForm[{Black}], 
   FaceForm[Lighter[Cyan]], Block[{s=Exp[I Pi/6.]},
   Triangle[{{-Re[s],-Im[s]},{Re[s],-Im[s]},{0,1}}]]}],.035},
  {Graphics[{EdgeForm[Black], FaceForm[Pink], 
   Polygon[{{-1,0},{0,1},{1,0},{0,-1}}]}],.04}
 };

(* The set of default options for plots and listplots *)
PlotAreaOptions = {
	Axes -> None,
	Frame -> True,
	PlotRange -> All,
	AspectRatio :> $AspectRatio,
	GridLines :> $GridLines,
	GridLinesStyle :> $GridLinesStyle,
	ImageSize :> $ImageSize,
	IntervalMarkers :> $IntervalMarkers,
	IntervalMarkersStyle :> $IntervalMarkersStyle,
	LabelStyle :> $LabelStyle,
	PlotMarkers :> $PlotMarkers,
	PlotStyle :> $PlotStyle
};

(* Modified default options for CurveFit data set plots *)
DataPlotAreaOptions = {
	PlotLayout->"Overlaid",
	PlotMarkers :> $DataPlotMarkers,
	PlotStyle :> $DataPlotStyle
};

(* Options unique to Data Plots *)
DataPlotOptions = {
	DataSizeThreshold :> $DataSizeThreshold,
	Tips :> $Tips
};
 


(***********************************************************)
(* Managing plot options usage messages *)

PlotAreaOptions::usage = 
 "\!\(\*StyleBox[\"PlotAreaOptions\",\nFontFamily->\"Courier\"]\): \n"<>
 "A variable that contains the list of general plot function default option assignment "<>
 "modifications applied during CurveFit package initialization. "<>
 "These are general plot options accepted by standard Mathematica "<>
 "functions such as Plot, ListPlot, etc., as well as CurveFit data plot "<>
 "functions, and these modified default rules are applied to both the "<>
 "standard functions and CurveFit's functions. Here are the "<>
 "modified rule assignments: "<>
 (PlotAreaOptions//Map[ToString]//Map["\n\t\t"<># &]//Apply[StringJoin])<>
 "\n\n"<>
 "Note that many of the assignments use RuleDelayed (\":>\") to "<>
 "associate an option "<>
 "with a variable that has the option's name prefixed by \"$\". "<>
 "Modifying the associated variable's value will affect the behavior of all "<>
 "subsequent plots. See the usage information for these variables for further help.";

DataPlotAreaOptions::usage = 
 "\!\(\*StyleBox[\"DataPlotAreaOptions\",\nFontFamily->\"Courier\"]\): \n"<>
 "A variable that contains a short list of plot function default option assignments "<>
 "applied during CurveFit package initialization. "<>
 "These plot option assignments are "<>
 "tailored for CurveFit data plots and override similar assignments "<>
 "specified by CurveFit's "<>
 "\!\(\*StyleBox[\"PlotAreaOptions\",\nFontFamily->\"Courier\"]\). "<>
 "Here are the "<>
 "modified rule assignments: "<>
 (DataPlotAreaOptions//Map[ToString]//Map["\n\t\t"<># &]//Apply[StringJoin])<>
 "\n\n"<>
 "Note the use of RuleDelayed (\":>\") to associate an option "<>
 "with a variable that has the option's name prefixed by \"$\". "<>
 "Modifying the associated variable's value will affect the behavior of all "<>
 "subsequent CurveFit data and fit plots."<>
 "\n\n"<>
 "See also: PlotAreaOptions, $DataPlotMarkers, $DataPlotStyle";

DataPlotOptions::usage = 
 "\!\(\*StyleBox[\"DataPlotOptions\",\nFontFamily->\"Courier\"]\): \n"<>
 "A variable that contains a short list of custom CurveFit plot options and "<>
 "their default option value assignments. "<>
 "These custom options are accepted only by CurveFit data and fit plot "<>
 "functions. Here are the options and their default assignments:\n"<>
 (DataPlotOptions//Map[ToString]//Map["\n\t\t"<># &]//Apply[StringJoin])<>
 "\n\n"<>
 "Note the use of RuleDelayed (\":>\") to associate an option "<>
 "with a variable that has the option's name prefixed by \"$\". "<>
 "Modifying the associated variable's value will affect the behavior of all "<>
 "subsequent CurveFit data and fit plots."<>
 "\n\n"<>
 "See also: DataSizeThreshold, Tips";

SetPlotAreaOptions::usage = 
 "TBD";




(***********************************************************)
(* Data and Fit Plot Function usage messages *)

LinearDataPlot::usage = 
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the active CurveFit data set using linear "<>
 "\!\(\*StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and "<>
 "\!\(\*StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes. "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)> "<>
 "represents a sequence of zero or more option expressions "<>
 "from the set of graphics option specifications accepted by ListPlot, plus some "<>
 "others specific to CurveFit. Any explicit option specification you provide, "<>
 "such as, for example, "<>
 "\!\(\*StyleBox[\"GridLines->None\",\nFontFamily->\"Courier\"]\), "<>
 "will override the default option behavior."<>
 "\n\n"<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"data\",\nFontSlant->\"Italic\"]\), "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the supplied CurveFit-formatted data set, "<>
 "such as a data set saved using "<>
 "\!\(\*StyleBox[\"BackupData\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)], "<>
 "so that "<>
 "\!\(\*StyleBox[\"LinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)] "<>
 "will plot the CurveFit data saved in "<>
 "\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)."<>
 "\n\n"<>
 "At initialization the CurveFit package changes several of Mathematica's standard "<>
 "plot option default values. These changes are specified in the usage messages for "<>
 "CurveFit's PlotAreaOptions, DataPlotAreaOptions, and DataPlotOptions."<>
 "\n\n"<>
 "See also the usage statements for:\n\t"<>
 "Tips, DataSizeThreshold, BackupData, LinearDifferencePlot";

LogDataPlot::usage = 
 "\!\(\*StyleBox[\"LogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the active CurveFit data set using linear "<>
 "\!\(\*StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and log "<>
 "\!\(\*StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes. "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)> "<>
 "represents a sequence of zero or more option expressions "<>
 "from the set of graphics option specifications accepted by ListPlot, plus some "<>
 "others specific to CurveFit. Any explicit option specification you provide, "<>
 "such as, for example, "<>
 "\!\(\*StyleBox[\"GridLines->None\",\nFontFamily->\"Courier\"]\), "<>
 "will override the default option behavior."<>
 "\n\n"<>
 "\!\(\*StyleBox[\"LogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"data\",\nFontSlant->\"Italic\"]\), "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the supplied CurveFit-formatted data set, "<>
 "such as a data set saved using "<>
 "\!\(\*StyleBox[\"BackupData\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)], "<>
 "so that "<>
 "\!\(\*StyleBox[\"LogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)] "<>
 "will plot the CurveFit data saved in "<>
 "\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)."<>
 "\n\n"<>
 "At initialization the CurveFit package changes several of Mathematica's standard "<>
 "plot option default values. These changes are specified in the usage messages for "<>
 "CurveFit's PlotAreaOptions, DataPlotAreaOptions, and DataPlotOptions."<>
 "\n\n"<>
 "See also the usage statements for:\n\t"<>
 "Tips, DataSizeThreshold, BackupData, LogDifferencePlot";

LogLogDataPlot::usage =
 "\!\(\*StyleBox[\"LogLogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the active CurveFit data set using log "<>
 "\!\(\*StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and "<>
 "\!\(\*StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes. "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)> "<>
 "represents a sequence of zero or more option expressions "<>
 "from the set of graphics option specifications accepted by ListPlot, plus some "<>
 "others specific to CurveFit. Any explicit option specification you provide, "<>
 "such as, for example, "<>
 "\!\(\*StyleBox[\"GridLines->None\",\nFontFamily->\"Courier\"]\), "<>
 "will override the default option behavior."<>
 "\n\n"<>
 "\!\(\*StyleBox[\"LogLogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"data\",\nFontSlant->\"Italic\"]\), "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the supplied CurveFit-formatted data set, "<>
 "such as a data set saved using "<>
 "\!\(\*StyleBox[\"BackupData\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)], "<>
 "so that "<>
 "\!\(\*StyleBox[\"LogLogDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)] "<>
 "will plot the CurveFit data saved in "<>
 "\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)."<>
 "\n\n"<>
 "At initialization the CurveFit package changes several of Mathematica's standard "<>
 "plot option default values. These changes are specified in the usage messages for "<>
 "CurveFit's PlotAreaOptions, DataPlotAreaOptions, and DataPlotOptions."<>
 "\n\n"<>
 "See also the usage statements for:\n\t"<>
 "Tips, DataSizeThreshold, BackupData, LogLogDifferencePlot";

LogLinearDataPlot::usage =
 "\!\(\*StyleBox[\"LogLinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the active CurveFit data set using log "<>
 "\!\(\*StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and linear "<>
 "\!\(\*StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes. "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)> "<>
 "represents a sequence of zero or more option expressions "<>
 "from the set of graphics option specifications accepted by ListPlot, plus some "<>
 "others specific to CurveFit. Any explicit option specification you provide, "<>
 "such as, for example, "<>
 "\!\(\*StyleBox[\"GridLines->None\",\nFontFamily->\"Courier\"]\), "<>
 "will override the default option behavior."<>
 "\n\n"<>
 "\!\(\*StyleBox[\"LogLinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"data\",\nFontSlant->\"Italic\"]\), "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the supplied CurveFit-formatted data set, "<>
 "such as a data set saved using "<>
 "\!\(\*StyleBox[\"BackupData\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)], "<>
 "so that "<>
 "\!\(\*StyleBox[\"LogLinearDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)] "<>
 "will plot the CurveFit data saved in "<>
 "\!\(\*StyleBox[\"var\",\nFontSlant->\"Italic\"]\)."<>
 "\n\n"<>
 "At initialization the CurveFit package changes several of Mathematica's standard "<>
 "plot option default values. These changes are specified in the usage messages for "<>
 "CurveFit's PlotAreaOptions, DataPlotAreaOptions, and DataPlotOptions."<>
 "\n\n"<>
 "See also the usage statements for:\n\t"<>
 "Tips, DataSizeThreshold, BackupData, LogLinearDifferencePlot";

ScatterDataPlot::usage =
 "\!\(\*StyleBox[\"ScatterDataPlot\",\nFontFamily->\"Courier\"]\)"<>
 "[<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>] "<>
 "plots the current CurveFit data as "<>
 "a scatter-plot of (x,y) points using linear "<>
 "\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes. "<>
 "By default, no error bars are displayed, and only simple dots are used to "<>
"represent the data points. "<>
 "<\!\(\*StyleBox[\"options\",\nFontSlant->\"Italic\"]\)> "<>
 "represents a sequence of zero or more option expressions "<>
 "from the set of graphics option specifications accepted by ListPlot, plus some "<>
 "others specific to CurveFit. Any explicit option specification you provide "<>
 "will override the default option behavior.";

FitPlot::usage = 
 "A graphics object containing the most recent plot of fit results "<>
 "generated by LinearDifferencePlot, etc.";
 
DiffPlot::usage = 
 "A graphics object containing the most recent plot of fit residuals "<>
 "generated by LinearDifferencePlot, etc.";

fitlogscales::usage =
 "The variable \!\(\*
StyleBox[\"fitlogscales\",\nFontWeight->\"Bold\"]\) contains a list of two booleans which indicate "<>
 "whether the plots FitPlot and DiffPlot were generated with "<>
 "log scales for the \!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axes (DiffPlot always uses a "<>
 "linear \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) axis, so only the first element of fitlogscales "<>
 "is relevant for DiffPlot).\n"<>
"\n"<>
"This information may be useful if you wish to "<>
"regenerate the results with different options using Show, e.g. "<>
"Show[FitPlot, <\!\(\*
StyleBox[\"options\",\nFontSlant->\"Italic\"]\)>]. Changing, for example, the PlotRange of a log-"<>
"scaled axis requires that the new range be specified using the natural logs of "<>
"the new limits. Assume you generated a fit plot using LogDifferencePlot, and you "<>
"want to replot this result over a \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) range of 30-1000. In this case,\n\t\t"<>
"fitlogscales = {False,True},\n  "<>
"confirming that the \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\)-axis is a log scale. "<>
"The correct graphing command would then be:\n\t\t"<>
"Show[ FitPlot, PlotRange \[Rule] Log[ {30,1000}] ]";

ClearFitPlots::usage = "Clears the values of FitPlot, DiffPlot, "<>
 "and fitlogscales.";

LinearDifferencePlot::usage =
 "LinearDifferencePlot[ ] "<>
 "Plots the best fit results in "<>
 "Lin(\!\(x\_i\)) - Lin(\!\(y\_i\)) form, and the difference "<>
 "plot in Lin(\!\(x\_i\)) - Lin(\!\(y\_i\)-y(\!\(x\_i\))) "<>
 "form. The argument list may include any graphics "<>
 "option specification accepted by ErrorListPlot or ListPlot.";

LogDifferencePlot::usage =
 "LogDifferencePlot[ ] "<>
 "Plots the best fit results in "<>
 "Lin(\!\(x\_i\)) - Log(\!\(y\_i\)) form, and the difference "<>
 "plot in Lin(\!\(x\_i\)) - Lin(\!\(y\_i\)-y(\!\(x\_i\))) "<>
 "form. The argument list may include any graphics "<>
 "option specification accepted by ErrorListPlot or ListPlot.";

LogLogDifferencePlot::usage =
 "LogLogDifferencePlot[ ] "<>
 "Plots the best fit results in "<>
 "Log(\!\(x\_i\)) - Log(\!\(y\_i\)) form, and the difference "<>
 "plot in Log(\!\(x\_i\)) - Lin(\!\(y\_i\)-y(\!\(x\_i\))) "<>
 "form. The argument list may include any graphics "<>
 "option specification accepted by ErrorListPlot or ListPlot.";

LogLinearDifferencePlot::usage =
 "LogLinearDifferencePlot[ ] "<>
 "Plots the best fit results in "<>
 "Log(\!\(x\_i\)) - Lin(\!\(y\_i\)) form, and the difference "<>
 "plot in Log(\!\(x\_i\)) - Lin(\!\(y\_i\)-y(\!\(x\_i\))) "<>
 "form. The argument list may include any graphics "<>
 "option specification accepted by ErrorListPlot or ListPlot.";

DifferencePlot::usage = 
 "A shared symbol for fit plot error messages.";
 
PlotInfo::usage = 
 "\!\(\*StyleBox[\"PlotInfo[ ]\",\nFontFamily->\"Courier\"]\) "<>
 "generates help information for CurveFit's Plotting functions and options."



Begin["`Private`"];

DataPlot::usage = "DataPlot[] is tha actual CurveFit data plotting function, except "<>
  "for ScatterDataPlot.";



(***********************************************************)
(*Plot Function Lists and SetPlotAreaOptions[]*)

PlotFunctionList = {
	(*Built-In Mathematica Functions*)
	Plot, LogPlot, LogLinearPlot, LogLogPlot, ParametricPlot
};

ListPlotFunctionList = {
	(*Built-In Mathematica Functions*)
	ListPlot, ListLogPlot, ListLogLinearPlot, ListLogLogPlot
};

DataPlotFunctionList = {
	(* CurveFit Functions *)
	LinearDataPlot, LogDataPlot, LogLinearDataPlot, LogLogDataPlot, ScatterDataPlot,
	DataPlot
};

FitPlotFunctionList = {
	(* CurveFit Functions *)
	LinearDifferencePlot, LogDifferencePlot, LogLinearDifferencePlot, LogLogDifferencePlot,
	DifferencePlot
};

(* Set the default options for the plotting routines *)
SetPlotAreaOptions[] := 
 Scan[
  SetOptions[#, Sequence @@ FilterRules[PlotAreaOptions, Options[Plot]]]&,
  PlotFunctionList
 ];
 Scan[
  SetOptions[#, Sequence @@ FilterRules[PlotAreaOptions, Options[ListPlot]]]&,
  ListPlotFunctionList
 ];
 Scan[
  (Options[#] = Join[DataPlotAreaOptions, FilterRules[Options[ListPlot],Except[DataPlotAreaOptions]],
   DataPlotOptions])&,
  Join[DataPlotFunctionList, FitPlotFunctionList]
 ];
 Scan[
  (AppendTo[Options[#], First[Options[Plot,PlotPoints]]])&,
  FitPlotFunctionList
 ];

SetPlotAreaOptions[];



(***********************************************************)
(* PlotDataFormat *)

PlotDataFormat[] = {
{xData_List, yData_List, xSigma_List, ySigma_List, nPoints_Integer},
name_String:None,
___}



(***********************************************************)
(* tiplist[]: get tips of the data point values for the plot *)

tiplist[f_/;MemberQ[ListPlotFunctionList, f], data_List]:= 
 Block[{plot},
 (* The Tooltip function call in the next line associates each data point {x,y} "#1" *)
 (* with the point's index "First[#2]" and its {x,y} value "#1", including sigmas   *)
  plot = f[MapIndexed[ Tooltip[#1,{First[#2],#1}]& , data], PlotMarkers -> None];
  Extract[plot,Position[plot,{__Tooltip}]][[1]]
 ]



(***********************************************************)
(* DataPlot : the function that does all the actual work *)

DataPlot[f_/;MemberQ[ListPlotFunctionList, f], opts:OptionsPattern[] ] :=
Block[
 {
   points = GetData["All"], 
   tips = OptionValue[Tips], 
   tooltips, 
   big = Map[n > # &, Sort[OptionValue[DataSizeThreshold]]], 
   style,
   label = OptionValue[PlotLabel],
   plot
 },
 If[!CheckLength[], Return[$Failed]];
 
 (* Title for the plot *)
 If[label === None, label = ExtractFileName[]];
 
 (* Generate Tooltip info for the points *)
 tooltips = If[tips === True || tips === All || (tips === Automatic && !Last[big]),
	tiplist[f, points],
	{} (* else no tooltips *)
	];
	
 (* Plot style *)
 style = { 
    PlotLabel -> label,
    If[First[big], PlotMarkers -> None, {}], (* Adjust for large data sets *)
    If[Last[big], 
      {IntervalMarkers -> None, Joined -> True},{}
    ] (* very large data sets *)
 };

  plot = f[ points, 
   FilterRules[{opts}, Options[f]], style, DataPlotAreaOptions
   ];
 AppendTo[ plot[[1]], {tooltips}]; (* add in the Tooltips *)
 plot
]

DataPlot[f_/;MemberQ[ListPlotFunctionList, f], d:PlotDataFormat[], opts:OptionsPattern[]] := 
Block[
 (* Use a Block to provide local values for the CurveFit internal data variables. *)
 (* This way we can just use the previous definition of DataPlot[].               *)
 {xx = xData, yy = yData,
  sx = xSigma, sy = ySigma,
  n = nPoints, DataFileName = name},
 DataPlot[f,opts]
 ]



(***********************************************************)
(* Specific DataPlots *)

LinearDataPlot[ opts:OptionsPattern[] ] := DataPlot[ListPlot, opts]
LinearDataPlot[d:PlotDataFormat[], opts:OptionsPattern[]] := DataPlot[ListPlot, d, opts]

LogDataPlot[ opts:OptionsPattern[] ] := DataPlot[ListLogPlot, opts]
LogDataPlot[d:PlotDataFormat[], opts:OptionsPattern[]] := DataPlot[ListLogPlot, d, opts]

LogLinearDataPlot[ opts:OptionsPattern[] ] := DataPlot[ListLogLinearPlot, opts]
LogLinearDataPlot[d:PlotDataFormat[], opts:OptionsPattern[]] := DataPlot[ListLogLinearPlot, d, opts]

LogLogDataPlot[ opts:OptionsPattern[] ] := DataPlot[ListLogLogPlot, opts]
LogLogDataPlot[d:PlotDataFormat[], opts:OptionsPattern[]] := DataPlot[ListLogLogPlot, d, opts]




(***********************************************************)
(* ScatterDataPlot : a special plotting function that doesn't use DataPlot *)

ScatterDataPlot[opts:OptionsPattern[] ] := 
Block[
 {
   points  = GetData["XY"], (* no uncertainties for this plot *)
   tips = OptionValue[Tips], 
   tooltips, 
   big = Map[n > # &, OptionValue[DataSizeThreshold]], 
   style,
   label = OptionValue[PlotLabel],
   plot
 },
 If[!CheckLength[], Return[$Failed]];
 
 (* Title for the plot *)
 If[label === None, label = ExtractFileName[]];
 
 (* Generate Tooltip info for the points *)
 tooltips = If[tips === True || tips === All || (tips === Automatic && !Last[big]),
	tiplist[ListPlot, points],
	{} (* else no tooltips *)
	];

(* Scatter plot style. Don't use Automatic AspectRatio if too oblong *)
style = {PlotMarkers -> None, PlotLabel -> label, Joined->False, 
	AspectRatio -> 
		Block[{x = points[[All,1]],y = points[[All,2]], dx, dy},
		  dx = Max[x]-Min[x]; dy=Max[y]-Min[y];
		  If[ dx != 0. && dy != 0. && (0.8 <= dx/dy <= 1/.8), Automatic, 1]
		]
};

plot = ListPlot[points, 
		FilterRules[{opts}, Options[ListPlot]], style, DataPlotAreaOptions];
AppendTo[ plot[[1]], {tooltips}]; (* add in the Tooltips *)
plot
];

ScatterDataPlot[d:PlotDataFormat[], opts:OptionsPattern[] ] := 
Block[
 (* Use a Block to provide local values for the CurveFit internal data variables. *)
 (* This way we can just use the previous definition of ScatterDataPlot[]. *)
 {xx = xData, yy = yData,
  sx = xSigma, sy = ySigma,
  n = nPoints, DataFileName = name},
 ScatterDataPlot[opts]
 ]



(***********************************************************)
(***********************************************************)
(* Plots of fit results *) 



(***********************************************************)
(* DifferencePlot : the function that does all the actual work *)

DifferencePlot[
    f_/;MemberQ[Most[FitPlotFunctionList], f],
	opts:OptionsPattern[]
   ] :=
Block[
 {
   Xvals, Yvals,  
   noPM = n > First[Sort[OptionValue[DataSizeThreshold]]],  (* If True, no PlotMarkers *)
   fYx,    (* holds fY[x] values and uncertainties *)
   noDiff, (* if True, couldn't propagate residual uncertainties *)
   Diff,   (* residual values: y - fY[x] *)
   sDiff,  (* residual ucertainties *)
   x,      (* independent variable for fY[x] *)
   xrange, (* plot domain for fY[x] *)
   fitp,   (* list plot function for data *)
   p,      (* plot function for fY[x] *)
   difp    (* list plot function for residuals *)
 },
 If[!CheckLength[], Return[$Failed]];
 If[OwnValues[fY]=={}, Message[DifferencePlot::uneval]; Return[$Failed]];

 (* generate the Plot[fY[x],..] domain argument: {x, min, max} *)
 xrange = Prepend[
            (* Through[] applies Min[] and Max[] to the data X values: *)
            Through[{Min,Max}[xx]],
            x (* prepends the x symbol *)
          ];

 (* select the various plot functions as determined by argument f *)
 Switch[f,
 LinearDifferencePlot,
	{fitp = LinearDataPlot, p = {Plot,Plot}, difp = LinearDataPlot, 
	fitlogscales = {False, False}},
 LogDifferencePlot,
	{fitp = LogDataPlot, p = {LogPlot,Plot}, difp = LinearDataPlot, 
	fitlogscales = {False, True}},
 LogLinearDifferencePlot,
	{fitp = LogLinearDataPlot, p = {LogLinearPlot,LogLinearPlot}, difp = LogLinearDataPlot, 
	fitlogscales  = {True, False}},
 LogLogDifferencePlot,
	{fitp = LogLogDataPlot, p = {LogLogPlot,LogLinearPlot}, difp = LogLinearDataPlot, 
	fitlogscales = {True, True}},
 True, Return[]
 ];

  FitPlot = 
	Show[
		fitp[opts, Joined -> False],
		First[p][
		   (* if noPM (no plot markers), uses the second plotstyle for the function plot *)
		   Evaluate[If[noPM, {Null,Tooltip[fY[x],funct]},Tooltip[fY[x],funct]]], 
		   Evaluate[xrange],
	       Evaluate[FilterRules[Join[{opts},DataPlotAreaOptions], Options[Plot]]]
	    ]
	];

(* 1.99f code:
 (* use Around[] to propagate errors and get a list of residuals with sigmas *)
   Xvals = Around @@@ Transpose[{xx,sx}]; 
   Yvals = Around @@@ Transpose[{yy,sy}]; 
   Diff  = List @@@ Transpose[Yvals - fY[Xvals]];  (* use Diff for temporary storage *)
   
 (* In the above line, Apply[List, Around[value, sigma]] \[Equal] {value,sigma} unless sigma \[Equal] 0 *)
 (* If a sigma is 0, then Around[value, 0] === value, just a number *)
 (* so that Apply[List, Around[value, 0]] \[Equal] value, not {value, 0} *)

 (* Given the above, we must find the Around[value, 0] cases to get {value, 0} for them *)
 (* We then transpose to get 2 vectors: Diff = residuals and  sDiff = sigmas) *)
 {Diff, sDiff} = Transpose[If[AtomQ[#],{#,0},#]& /@ Diff]; 
*)
 
  Xvals = Around @@@ Transpose[{xx,sx}]; 
  Yvals = Around @@@ Transpose[{yy,sy}]; 
   
 (* Attempt to use Around[] to propagate fY[x] errors *)
 fYx = fY[Xvals];
 If[!AllTrue[fYx, (NumberQ[#] || Head[#]===Around)&],
     If[AllTrue[(fYx = fY'[xx]sx),NumberQ],
        fYx = Around @@@ Transpose[{fY[xx],fYx}],
        fYx = fY[xx]; Message[DifferencePlot::fYerrors]
      ]
 ];
 Diff  = List @@@ Transpose[Yvals - fYx];  (* use Diff for temporary storage *)
   
 (* In the above line, Apply[List, Around[value, sigma]] \[Equal] {value,sigma} unless sigma \[Equal] 0 *)
 (* If a sigma is 0, then Around[value, 0] === value, just a number *)
 (* so that Apply[List, Around[value, 0]] \[Equal] value, not {value, 0} *)

 (* Given the above, we must find the Around[value, 0] cases to get {value, 0} for them *)
 (* We then transpose to get 2 vectors: Diff = residuals and  sDiff = sigmas) *)
 {Diff, sDiff} = Transpose[If[AtomQ[#],{#,0},#]& /@ Diff]; 



 DiffPlot = 
	Show[
		difp[{{xx, Diff, 0 sx, sDiff, n},"Residuals and Effective Uncertainties"}, 
		  opts, Joined -> False],
		Last[p][
		   Tooltip[0,funct], 
		   Evaluate[xrange],
		   PlotStyle -> {{Directive[Black,Thick,Dashed]}},
	       Evaluate[FilterRules[Join[{opts},DataPlotAreaOptions], Options[Plot]]]
	    ]
	];

 Column[{
	FitPlot,
	DiffPlot,
	Framed[funct,FrameMargins->10], 
	Framed[results,FrameMargins->10]}]
]; (* end definition of DifferencePlot[f,opts] *)

DifferencePlot[f_/;MemberQ[Most[FitPlotFunctionList], f], 
   d:PlotDataFormat[], opts:OptionsPattern[]] := 
Block[
 (* Use a Block to provide local values for the CurveFit internal data variables. *)
 (* This way we can just use the previous definition of DataPlot[].               *)
 {xx = xData, yy = yData,
  sx = xSigma, sy = ySigma,
  n = nPoints, DataFileName = name},
 DifferencePlot[f,opts]
 ]

(***********************************************************)
(* Error messages *)

DifferencePlot::uneval = 
"First fit the data, then plot the fit results.";

DifferencePlot::fYerrors = 
"Can't propagate uncertainties in all fY[x] calculations; x uncertainties not included.";



(***********************************************************)
(* Specific Difference Plots *)

LinearDifferencePlot[ opts:OptionsPattern[] ] := DifferencePlot[LinearDifferencePlot, opts]
LinearDifferencePlot[d:PlotDataFormat[], opts:OptionsPattern[]] := 
  DifferencePlot[LinearDifferencePlot, d, opts]

LogDifferencePlot[ opts:OptionsPattern[] ] := DifferencePlot[LogDifferencePlot, opts]
LogDifferencePlot[d:PlotDataFormat[], opts:OptionsPattern[]] := 
  DifferencePlot[LogDifferencePlot, d, opts]

LogLinearDifferencePlot[ opts:OptionsPattern[] ] := DifferencePlot[LogLinearDifferencePlot, opts]
LogLinearDifferencePlot[d:PlotDataFormat[], opts:OptionsPattern[]] := 
  DifferencePlot[LogLinearDifferencePlot, d, opts]

LogLogDifferencePlot[ opts:OptionsPattern[] ] := DifferencePlot[LogLogDifferencePlot, opts]
LogLogDifferencePlot[d:PlotDataFormat[], opts:OptionsPattern[]] := 
  DifferencePlot[LogLogDifferencePlot, d, opts]




(***********************************************************)
(* ClearFitPlots *)

ClearFitPlots := Clear[FitPlot, DiffPlot, fitlogscales];


(***********************************************************)
(***********************************************************)
(* PlotInfo[]: help Info for Plots *)

(* Names of the CurveFit Data Plotting functions as String alternatives *)
DataPlotNames = 
 DataPlotFunctionList//Most//Map[ToString]//
   Apply[Alternatives]//StringExpression;

(* Names of the CurveFit Data Plotting functions as String alternatives *)
FitPlotNames = 
 FitPlotFunctionList//Map[ToString]//
   Apply[Alternatives]//StringExpression;

(* Names of the CurveFit-modified plot options as String alternatives *)
PlotOptions = 
 Union[PlotAreaOptions,DataPlotAreaOptions, DataPlotOptions]//
   Map[Apply[List]]//Sort//Transpose//First//Map[ToString]//
   Apply[Alternatives]//StringExpression;

(* Names of the CurveFit-assigned plot default option values as String alternatives *)
PlotDefaults = 
{"$AspectRatio", "$DataPlotMarkers", "$DataPlotStyle", "$DataSizeThreshold", 
 "$GridLines", "$GridLinesStyle", "$ImageSize", "$IntervalMarkers", 
 "$IntervalMarkersStyle", "$LabelStyle", "$PlotMarkers", "$PlotStyle", "$Tips"}//
   Apply[Alternatives]//StringExpression;
   
PlotInfo[] :=
 Column[{
	Style["CurveFit Data Plotting Functions:", Bold, 12],
	Information[DataPlotNames, LongForm->False],
	Null,
	Style["CurveFit Fit Results Plotting Functions:",Bold,12],
	Information[FitPlotNames, LongForm->False],
	Null,
	Style["General information about Graphics options settings:",Bold,12],
	Information[
	  StringExpression["PlotAreaOptions"|"DataPlotAreaOptions"|"DataPlotOptions"], 
	  LongForm->False],
	Null,
	Style["Graphics options whose default values are set by CurveFit:",Bold,12],
	Information[PlotOptions, LongForm->False],
	Null,
	Style["Plot options' default values:",Bold,12],
	Information[PlotDefaults, LongForm->False]
 }]



(***********************************************************)
End[]; (* `Private` *)

