(* ::Package:: *)

(* Copyright 1997-2021 by California Inst. of Technology, Pasadena, CA. *)

(* Only call this package file from CurveFit.m *)
(* The code assumes that we are already in the CurveFit` context. *)

(* Fit.m - Fitting routine definitions *)

(* 
Each predefined fitting function has two routines:
   <function>Fit and f<function>[], where <function> is the name
	of the fitting function, such as: SlopeFit and fSlope[].

The <function>Fit routine performs chi-squared minimization to
determine the best-fit parameter values, and sets the values
of the several fitting parameters it uses to the values found.

The f<function>[] routine defines a function of one variable to
calculate y[x] based on either the best-fit parameter values or
parameter values supplied by the user. 
*)

(*
This file includes the common definitions for fitting parameters
and then includes the several files which contain the fit code.
*)


(***********************************************************)
(* Private variables used to share data among routines *)

Begin["`Private`"];

chiplus::usage = "Used in parameter sigma calculations.";

(* generated by fitting routines, used by plots of fit results *)
funct::usage = "The function expression for the fit.";
yff::usage = "yff[x] is the fit defined by a fitting function.";
ssy::usage = "A list of the effective uncertainties (including sx).";

End[]; (* `Private` *)



(***********************************************************)
(* usages *)

(* a categorized list of the names of the fitting functions *)
FitList::usage = "A categorized list of the names of the "<>
"fitting functions used to generate the CurveFit palette menu "<>
" of fits.";

(* results from the latest fit *)
FitResults::usage = "Gives a formatted printout of "<>
"the parameter values, sigmas, and \!\(\*SuperscriptBox[\"\[Chi]\", \"2\"]\) from the latest fit to the "<>
" data.";

(* The function which results from the latest fit *)
fY::usage = "The function fY[x] is the optimized result of the latest "<>
"fit to the data.";
fYstring::usage = "A string representation of the latest fit function y(x).";
Pnames::usage = "A list of the latest fit's parameter symbol name Strings.";
Signames::usage = "A list of the latest fit's parameter sigma symbol name Strings.";
(* The number of data points and the number of parameters used during the latest fit *)
Nfit::usage = "The number of data points used during the latest fit.";
Pfit::usage = "The number of parameters optimized during the latest fit.";
Tfit::usage = "A String describing what uncertainties were used during the latest fit.";

(* Calculate the effective variances in the fit residuals *)
ResidualVariances::usage = "ResidualVariances[\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)] returns the effective, full variances of the "<>
"current CurveFit data set residuals using both \!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) uncertainties: \!\(\*SuperscriptBox[\(\[Sigma]\), \(2\)]\)= \!\(\*SubsuperscriptBox[\(\[Sigma]\), \(y\), \(2\)]\)+(\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"'\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"[\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"]\",\nFontSlant->\"Italic\"]\)\!\(\*SubscriptBox[\(\[Sigma]\), \(x\)]\)\!\(\*SuperscriptBox[\()\), \(2\)]\). "<>
"If the argument is omitted, it uses the latest fit result function, fY.";

(* Compare a fitting function to the current data set and calculate Residuals *)
Residuals::usage = "Residuals[\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)] calculates a list of \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\)-\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)[\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)] values for the function named \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\) over "<>
"the \!\(\*
StyleBox[\"CurveFit\",\nFontSlant->\"Italic\"]\) data set. If no argument is given, then the latest fit result (fY) is used for \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\).";

NormedResiduals::usage = "NormedResiduals[\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\)] calculates a list of (\!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\)-\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)[\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)])/\[Sigma] values for the function named \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\) over "<>
"the \!\(\*
StyleBox[\"CurveFit\",\nFontSlant->\"Italic\"]\) data set (with \!\(\*
StyleBox[\"n\",\nFontSlant->\"Italic\"]\) data points). The argument \!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\) gives the number of optimized parameters in \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)[\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)], and it "<>
"defaults to 0 if unspecified. Each ratio is further divided by the square-root of (1-\!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"/\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"n\",\nFontSlant->\"Italic\"]\)), so that "<>
"the expected value of its square is equal to 1. This means that these normalized residuals "<>
"for the correct \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)[\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)] and Gaussian "<>
"errors should be samples of a Gaussian distribution with a standard deviation of 1. Note that the residual uncertainties "<>
"are \!\(\*
StyleBox[\"generalized\",\nFontSlant->\"Italic\"]\) using both \!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and \!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\) uncertainties (see ResidualVariances). If the generalized uncertainty for any (x,y) "<>
"data point is undefined or vanishes, then the function divides all residuals by an unbiased standard deviation estimate "<>
"(which would make the fit's \!\(\*
StyleBox[\"reduced\",\nFontSlant->\"Italic\"]\) \[Chi]-\!\(\*
StyleBox[\"squared\",\nFontSlant->\"Italic\"]\) equal 1).\n\n"<>
"NormedResiduals[] uses the latest fit result for \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\) and \!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\).";

(* Compare a fitting function to the current data set and calculate ChiSquared *)
ChiSquared::usage = "ChiSquared[\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\)] calculates the \[Chi]-\!\(\*
StyleBox[\"squared\",\nFontSlant->\"Italic\"]\) value for the function named \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\) over "<>
"the \!\(\*
StyleBox[\"CurveFit\",\nFontSlant->\"Italic\"]\) data set.\n\n"<>
"ChiSquared[\!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\),\!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\)] calculates the \!\(\*
StyleBox[\"reduced\",\nFontSlant->\"Italic\"]\) \[Chi]-\!\(\*
StyleBox[\"squared\",\nFontSlant->\"Italic\"]\) value for the function named \!\(\*
StyleBox[\"f\",\nFontSlant->\"Italic\"]\) that was fit to "<>
"the data using \!\(\*
StyleBox[\"p\",\nFontSlant->\"Italic\"]\) parameters.\n\n"<>
"ChiSquared[] (no argument) uses the latest fit result for the function and returns a \!\(\*
StyleBox[\"reduced\",\nFontSlant->\"Italic\"]\) "<>
"\[Chi]-\!\(\*
StyleBox[\"squared\",\nFontSlant->\"Italic\"]\) value over the current data set (not necessarily the data set used when fitting).\n\n"
"Note that the value returned is a \!\(\*
StyleBox[\"generalized\",\nFontSlant->\"Italic\"]\) \[Chi]-\!\(\*
StyleBox[\"squared\",\nFontSlant->\"Italic\"]\) based on residual variances "<>
"normalized by using both x and y uncertainties (see ResidualVariances).\n\n If data uncertainties "<>
"are not supplied, or any one data "<>
"point's uncertainty vanishes, the function reports the error and just returns the integer value 0";

(* Clear all fit information *)
ClearFit::usage="Clears the values of all fit parameters, fY, and fit plots.";

(* The parameters used for fitting functions; *)
(* set by the fitting routines *)
a::usage =
b::usage =
c::usage =
d::usage =
e::usage =
gamma::usage =
omega0::usage =
ymax::usage =
tau::usage =
Q::usage =
mean::usage =
sigma::usage =
xMinimum::usage =
theta0::usage =
"A parameter variable set by a fitting routine to "<>
"best fit the data.";

siga::usage =
sigb::usage =
sigc::usage =
sigd::usage =
sige::usage =
sigg::usage =
sigo::usage =
sigy::usage =
sigtau::usage =
sigQ::usage =
sigmean::usage =
sigsigma::usage =
sigt::usage =
"The uncertainty in a parameter value found by a "<>
"fitting routine";

(* These parameters must be set by the user; *)
(* needed for some fitting routines *)
region1;
region2;



(***********************************************************)
(* Error messages *)

ChiSquared::badsigma = "At least one data point has an undefined or 0 total uncertainty.";



Begin["`Private`"];



(***********************************************************)
(* ClearFitParameters *)

ClearFit := 
(Clear[
CurveFit`a,
CurveFit`b,
CurveFit`c,
CurveFit`d,
CurveFit`e,
CurveFit`gamma,
CurveFit`omega0,
CurveFit`ymax,
CurveFit`tau,
CurveFit`Q,
CurveFit`mean,
CurveFit`sigma,
CurveFit`xMinimum,
CurveFit`theta0,
CurveFit`siga,
CurveFit`sigb,
CurveFit`sigc,
CurveFit`sigd,
CurveFit`sige,
CurveFit`sigg,
CurveFit`sigo,
CurveFit`sigy,
CurveFit`sigtau,
CurveFit`sigQ,
CurveFit`sigmean,
CurveFit`sigsigma,
CurveFit`sigt,
CurveFit`fY,
CurveFit`Nfit,
CurveFit`Pfit,
CurveFit`Tfit,
CurveFit`Signames,
CurveFit`Pnames,
CurveFit`Private`chiplus,
CurveFit`Private`funct,
CurveFit`Private`results,
CurveFit`Private`yff,
CurveFit`Private`ssy
]; CurveFit`ClearFitPlots;
);

ClearFit;



(***********************************************************)
(* FitResults, choices of Tfit strings *)

FitResults := CurveFit`Private`results

fYstring := CurveFit`Private`funct

TfitStrings = {
	Style["Fit of (x,y) (unweighted):",Bold,Italic],
	Style["Fit of (x,y\[PlusMinus]\!\(\*SubscriptBox[\(\[Sigma]\), \(y\)]\)):",Bold,Italic],
	Style["Fit of (x\[PlusMinus]\!\(\*SubscriptBox[\(\[Sigma]\), \(x\)]\),y\[PlusMinus]\!\(\*SubscriptBox[\(\[Sigma]\), \(y\)]\)):",Bold,Italic]
	};



(***********************************************************)
(* ResidualVariances, Residuals, ChiSquared *)

ResidualVariances[f_:fY]:=(
	If[!CurveFit`CheckLength[], Abort[]];
	Block[{ss = N[CurveFit`sy]^2},
		If[Min[CurveFit`sx] > 0, ss += (Thread[(f)'[N[CurveFit`xx]]]CurveFit`sx)^2];
		ss
	]
)

Residuals[f_:fY]:=(
	If[!CurveFit`CheckLength[], Abort[]];
	N[CurveFit`yy] - (f)/@(N[CurveFit`xx])
)

NormedResiduals[f_,p_:0]:=(
	If[!CurveFit`CheckLength[], Abort[]];
	Block[{rr = Residuals[f], ss = ResidualVariances[f], norm = 1/Sqrt[1.-p/CurveFit`n]},
		If[(Min[ss] > 0), norm rr/Sqrt[ss], rr Sqrt[CurveFit`n/(Plus@@(rr^2))] ]
	]
)
NormedResiduals[]:= NormedResiduals[fY,Pfit]

ChiSquared[f_]:=(
	If[!CurveFit`CheckLength[], Abort[]];
	Block[{rr = Residuals[f]^2, ss = ResidualVariances[f]},
		If[!(Min[ss] > 0), Message[ChiSquared::badsigma]; Return[0]];
		Plus@@(rr/ss)
	]
)

ChiSquared[f_,p_]:=(ChiSquared[f])/(CurveFit`n-p)

ChiSquared[]:=ChiSquared[fY,Pfit]



(***********************************************************)
(* findsig[]. Used by the fitting routines *)

(* FINDING THE SIGMA GIVEN THE VALUE OF THE VARIBLE AND
 * ITS ROUGH SIGMA
 * In order to find siga, we first find the lower bound by minimizing
 * (chi[a+x,b]-chiplus)^2, (i.e. minimizing chi with the other
 * variables fixed, and not reminimized at each a+x. )
 * From that rsa (rough-sigma-a), we then move in steps of 10, 
 * i.e. rsa*10, rsa*100, etc. till fa[a+rsa*10^i] > chiplus.
 * now we found the order of magnitude of siga. (i.e. rsa is within 
 * a factor of 10 of the actual siga.)
 * Now, we proceed with steps of rsa. i.e. rsa, rsa*2, rsa*3... etc.
 * till we again stop at fa[a+rsa*i] > chiplus.
 * Now we have bracketed siga with a range of width rsa.
 * (e.g. siga is within 4*rsa and 5*rsa.)
 * So now we take the 3 points: 4*rsa, 4.5*rsa, 5*rsa and we fit a
 * parabola through them, we extrapolate where it intercepts chiplus
 * and we have our siga. This procedure gives siga wit accuracy of
 * at least  2 significant digits which is actually all we want. 
 * Test with the code which uses explicitly the routine FindRoot to solve 
 * for x: fa[a+x]=chiplus, gave the same errors for the parameters in all
 * the data sets tried.
 *)

findsig[a_,rsaa_,fa_]:=Block[{rs,x1,x2,x3,f1,f2,f3,e1,e2,e3},
rs=Sign[a]*rsaa/10;
Do[rs=rs*10;If[fa[a+rs*10] >  chiplus,Break[]],{i,20}];
f3=fa[a+rs];
Do[f1=f3;If[ (f3=fa[a+rs*i]) > chiplus,
x1=rs*(i-1);
x2=rs*(i-.5);
x3=rs*i;
f2=fa[a+x2];
e1=(f3*x1*(x1-x2)*x2+x3*(f1*x2*(x2-x3)+f2*x1*(-x1+x3)))/
((x1-x2)*(x1-x3)*(x2-x3));
e2=(f3*(-x1^2+x2^2)+f2*(x1^2-x3^2)+f1*(-x2^2+x3^2))/
((x1-x2)*(x1-x3)*(x2-x3));
e3=(f3*(x1-x2)+f1*(x2-x3)+f2*(-x1+x3))/
((x1-x2)*(x1-x3)*(x2-x3));
Break[];],{i,2,10}];
Abs[(-e2+Sign[a]*Sqrt[e2^2+4*e3*(chiplus-e1)])/2/e3]
];



(***********************************************************)
End[]; (* `Private` *)



(***********************************************************)
(* Load the fit function definitions *)

FitList={};
(* If you change the Get position of Fit4.m, then you must change the 
    code in the palette file fits.m *)
Get[ToFileName[{"CurveFit","FitFiles"}, "Fit1.m"]];
Get[ToFileName[{"CurveFit","FitFiles"}, "Fit4.m"]];
Get[ToFileName[{"CurveFit","FitFiles"}, "Fit3.m"]];
Get[ToFileName[{"CurveFit","FitFiles"}, "Fit2.m"]];

